/** @file
  Functions implementation for Bus Specific Driver Override protoocl.

Copyright (c) 2006 - 2017, Intel Corporation. All rights reserved.<BR>
This program and the accompanying materials
are licensed and made available under the terms and conditions of the BSD License
which accompanies this distribution.  The full text of the license may be found at
http://opensource.org/licenses/bsd-license.php

THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS,
WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.

**/

#include "PciBus.h"

/**
  Initializes a PCI Driver Override Instance.

  @param  PciIoDevice   PCI Device instance.

**/
VOID
InitializePciDriverOverrideInstance (
  IN OUT PCI_IO_DEVICE          *PciIoDevice
  )
{
  PciIoDevice->PciDriverOverride.GetDriver = GetDriver;
}

//
// ROYAL_PARK_OVERRIDE: RoyalParkOverrideBegin - RPPO-KBL-0087
//
/**
  Find the image handle whose path equals to ImagePath.

  @param ImagePath   Image path.

  @return Image handle.
**/
EFI_HANDLE
LocateImageHandle (
  IN EFI_DEVICE_PATH_PROTOCOL   *ImagePath
  )
{
  EFI_STATUS                    Status;
  EFI_HANDLE                    *Handles;
  UINTN                         Index;
  UINTN                         HandleNum;
  EFI_DEVICE_PATH_PROTOCOL      *DevicePath;
  UINTN                         ImagePathSize;
  EFI_HANDLE                    ImageHandle;

  Status = gBS->LocateHandleBuffer (
                  ByProtocol,
                  &gEfiLoadedImageDevicePathProtocolGuid,
                  NULL,
                  &HandleNum,
                  &Handles
                  );
  if (EFI_ERROR (Status)) {
    return NULL;
  }

  ImageHandle   = NULL;
  ImagePathSize = GetDevicePathSize (ImagePath);

  for (Index = 0; Index < HandleNum; Index++) {
    Status = gBS->HandleProtocol (Handles[Index], &gEfiLoadedImageDevicePathProtocolGuid, (VOID **) &DevicePath);
    if (EFI_ERROR (Status)) {
      continue;
    }
    if ((ImagePathSize == GetDevicePathSize (DevicePath)) &&
        (CompareMem (ImagePath, DevicePath, ImagePathSize) == 0)
        ) {
      ImageHandle = Handles[Index];
      break;
    }
  }

  FreePool (Handles);
  return ImageHandle;
}
//
// ROYAL_PARK_OVERRIDE: RoyalParkOverrideEnd - RPPO-KBL-0087
//

/**
  Uses a bus specific algorithm to retrieve a driver image handle for a controller.

  @param  This                  A pointer to the EFI_BUS_SPECIFIC_DRIVER_OVERRIDE_PROTOCOL instance.
  @param  DriverImageHandle     On input, a pointer to the previous driver image handle returned
                                by GetDriver(). On output, a pointer to the next driver
                                image handle. Passing in a NULL, will return the first driver
                                image handle.

  @retval EFI_SUCCESS           A bus specific override driver is returned in DriverImageHandle.
  @retval EFI_NOT_FOUND         The end of the list of override drivers was reached.
                                A bus specific override driver is not returned in DriverImageHandle.
  @retval EFI_INVALID_PARAMETER DriverImageHandle is not a handle that was returned on a
                                previous call to GetDriver().

**/
EFI_STATUS
EFIAPI
GetDriver (
  IN EFI_BUS_SPECIFIC_DRIVER_OVERRIDE_PROTOCOL              *This,
  IN OUT EFI_HANDLE                                         *DriverImageHandle
  )
{
//
// ROYAL_PARK_OVERRIDE: RoyalParkOverrideBegin - RPPO-KBL-0087
//
//  PCI_IO_DEVICE             *PciIoDevice;
//  LIST_ENTRY                *CurrentLink;
//  PCI_DRIVER_OVERRIDE_LIST  *Node;
//
//  PciIoDevice = PCI_IO_DEVICE_FROM_PCI_DRIVER_OVERRIDE_THIS (This);
//
//  CurrentLink = PciIoDevice->OptionRomDriverList.ForwardLink;
//
//  while (CurrentLink != NULL && CurrentLink != &PciIoDevice->OptionRomDriverList) {
//
//    Node = DRIVER_OVERRIDE_FROM_LINK (CurrentLink);
//
//    if (*DriverImageHandle == NULL) {
//
//      *DriverImageHandle = Node->DriverImageHandle;
//      return EFI_SUCCESS;
//    }
//
//    if (*DriverImageHandle == Node->DriverImageHandle) {
//
//      if (CurrentLink->ForwardLink == &PciIoDevice->OptionRomDriverList ||
//          CurrentLink->ForwardLink == NULL) {
//        return EFI_NOT_FOUND;
//      }
//
//      //
//      // Get next node
//      //
//      Node                = DRIVER_OVERRIDE_FROM_LINK (CurrentLink->ForwardLink);
//      *DriverImageHandle  = Node->DriverImageHandle;
//      return EFI_SUCCESS;
//    }
//
//    CurrentLink = CurrentLink->ForwardLink;
//  }
//
//  return EFI_INVALID_PARAMETER;
  PCI_IO_DEVICE             *PciIoDevice;
  LIST_ENTRY                *Link;
  PCI_DRIVER_OVERRIDE_LIST  *Override;
  BOOLEAN                   ReturnNext;

  Override    = NULL;
  PciIoDevice = PCI_IO_DEVICE_FROM_PCI_DRIVER_OVERRIDE_THIS (This);
  ReturnNext  = (BOOLEAN) (*DriverImageHandle == NULL);
  for ( Link = GetFirstNode (&PciIoDevice->OptionRomDriverList)
      ; !IsNull (&PciIoDevice->OptionRomDriverList, Link)
      ; Link = GetNextNode (&PciIoDevice->OptionRomDriverList, Link)
      ) {

    Override = DRIVER_OVERRIDE_FROM_LINK (Link);

    if (ReturnNext) {
      if (Override->DriverImageHandle == NULL) {
        Override->DriverImageHandle = LocateImageHandle (Override->DriverImagePath);
      }

      if (Override->DriverImageHandle == NULL) {
        //
        // The Option ROM identified by Override->DriverImagePath is not loaded.
        //
        continue;
      } else {
        *DriverImageHandle = Override->DriverImageHandle;
        return EFI_SUCCESS;
      }
    }

    if (*DriverImageHandle == Override->DriverImageHandle) {
      ReturnNext = TRUE;
    }
  }

  ASSERT (IsNull (&PciIoDevice->OptionRomDriverList, Link));
  //
  // ReturnNext indicates a handle match happens.
  // If all nodes are checked without handle match happening,
  // the DriverImageHandle should be a invalid handle.
  //
  if (ReturnNext) {
    return EFI_NOT_FOUND;
  } else {
    return EFI_INVALID_PARAMETER;
  }
//
// ROYAL_PARK_OVERRIDE: RoyalParkOverrideEnd - RPPO-KBL-0087
//
}

/**
  Add an overriding driver image.

  @param PciIoDevice        Instance of PciIo device.
  @param DriverImageHandle  Image handle of newly added driver image. // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  @param DriverImagePath    Device path of newly added driver image.  // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

  @retval EFI_SUCCESS          Successfully added driver.
  @retval EFI_OUT_OF_RESOURCES No memory resource for new driver instance.
  @retval other                Some error occurred when locating gEfiLoadedImageProtocolGuid.

**/
EFI_STATUS
AddDriver (
  IN PCI_IO_DEVICE            *PciIoDevice,      // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  IN EFI_HANDLE               DriverImageHandle, // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  IN EFI_DEVICE_PATH_PROTOCOL *DriverImagePath   // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  )
{
  //EFI_STATUS                    Status;       // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  //EFI_LOADED_IMAGE_PROTOCOL     *LoadedImage; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  //PE_COFF_LOADER_IMAGE_CONTEXT  ImageContext; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  PCI_DRIVER_OVERRIDE_LIST      *Node;

//
// ROYAL_PARK_OVERRIDE: RoyalParkOverrideBegin - RPPO-KBL-0087
//
//  Status = gBS->HandleProtocol (DriverImageHandle, &gEfiLoadedImageProtocolGuid, (VOID **) &LoadedImage);
//  if (EFI_ERROR (Status)) {
//    return Status;
//  }
  //
  // Caller should pass in either Image Handle or Image Path, but not both.
  //
  ASSERT ((DriverImageHandle == NULL) || (DriverImagePath == NULL));
//
// ROYAL_PARK_OVERRIDE: RoyalParkOverrideEnd - RPPO-KBL-0087
//

  Node = AllocateZeroPool (sizeof (PCI_DRIVER_OVERRIDE_LIST)); // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  if (Node == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  Node->Signature         = DRIVER_OVERRIDE_SIGNATURE;
  Node->DriverImageHandle = DriverImageHandle;
  Node->DriverImagePath   = DuplicateDevicePath (DriverImagePath); // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

  InsertTailList (&PciIoDevice->OptionRomDriverList, &Node->Link); // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

  PciIoDevice->BusOverride  = TRUE;

  //ImageContext.Handle    = LoadedImage->ImageBase;          // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  //ImageContext.ImageRead = PeCoffLoaderImageReadFromMemory; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

  //
  // Get information about the image
  //
  //PeCoffLoaderGetImageInfo (&ImageContext); // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

  return EFI_SUCCESS;
}

